# Voter List API

A Laravel-based RESTful API for a mobile voter list application with user registration, admin approval system, and role-based access control.

## Features

- ✅ User Registration with email, phone, and password
- ✅ Admin approval system for new users
- ✅ Role-based access (Admin and User)
- ✅ JWT-based authentication using Laravel Sanctum
- ✅ Secure password hashing
- ✅ RESTful API endpoints
- ✅ MySQL database integration

## Requirements

- PHP >= 8.2
- Composer
- MySQL
- Laravel 12.x

## Installation

### 1. Clone or navigate to the project directory

```bash
cd "d:\xampp\htdocs\New folder\voter-list-api"
```

### 2. Install dependencies

```bash
composer install
```

### 3. Configure environment

The `.env` file is already configured with MySQL settings:

- Database: `voter_list`
- Username: `root`
- Password: (empty for XAMPP default)

### 4. Create the database

Open MySQL/phpMyAdmin and create a database named `voter_list`:

```sql
CREATE DATABASE voter_list;
```

### 5. Generate application key

```bash
php artisan key:generate
```

### 6. Run migrations

```bash
php artisan migrate
```

### 7. Seed the database (Optional - creates default admin user)

```bash
php artisan db:seed
```

This creates:

- **Admin User**:
    - Email: `admin@voterlist.com`
    - Phone: `+1234567890`
    - Password: `admin123`
    - Role: `admin`
    - Approved: `true`

- **Sample User**:
    - Email: `user@example.com`
    - Phone: `+9876543210`
    - Password: `user123`
    - Role: `user`
    - Approved: `false`

### 8. Start the development server

```bash
php artisan serve
```

The API will be available at: `http://localhost:8000`

## API Endpoints

### Base URL

```
http://localhost:8000/api
```

### Public Endpoints

#### 1. User Registration

```http
POST /api/register
Content-Type: application/json

{
    "email": "user@example.com",
    "phone": "+1234567890",
    "password": "password123",
    "password_confirmation": "password123"
}
```

**Response (201 Created):**

```json
{
    "success": true,
    "message": "Registration successful. Please wait for admin approval.",
    "data": {
        "id": 1,
        "email": "user@example.com",
        "phone": "+1234567890",
        "role": "user",
        "is_approved": false
    }
}
```

#### 2. User Login

```http
POST /api/login
Content-Type: application/json

{
    "email": "admin@voterlist.com",
    "password": "admin123"
}
```

**Response (200 OK):**

```json
{
    "success": true,
    "message": "Login successful",
    "data": {
        "user": {
            "id": 1,
            "email": "admin@voterlist.com",
            "phone": "+1234567890",
            "role": "admin",
            "is_approved": true
        },
        "token": "1|laravel_sanctum_token_here..."
    }
}
```

**Note:** If user is not approved:

```json
{
    "success": false,
    "message": "Your account is not approved yet. Please wait for admin approval."
}
```

### Protected Endpoints (Require Authentication Token)

**Add this header to all protected requests:**

```
Authorization: Bearer {your_token_here}
```

#### 3. Get Current User Info

```http
GET /api/me
Authorization: Bearer {token}
```

**Response:**

```json
{
    "success": true,
    "data": {
        "id": 1,
        "email": "admin@voterlist.com",
        "phone": "+1234567890",
        "role": "admin",
        "is_approved": true,
        "created_at": "2026-01-31T10:00:00.000000Z"
    }
}
```

#### 4. Logout

```http
POST /api/logout
Authorization: Bearer {token}
```

**Response:**

```json
{
    "success": true,
    "message": "Logged out successfully"
}
```

### Admin Endpoints (Require Admin Role)

#### 5. Get All Users

```http
GET /api/admin/users
Authorization: Bearer {admin_token}
```

**Response:**

```json
{
    "success": true,
    "data": [
        {
            "id": 1,
            "email": "admin@voterlist.com",
            "phone": "+1234567890",
            "role": "admin",
            "is_approved": true,
            "created_at": "2026-01-31T10:00:00.000000Z"
        },
        {
            "id": 2,
            "email": "user@example.com",
            "phone": "+9876543210",
            "role": "user",
            "is_approved": false,
            "created_at": "2026-01-31T10:05:00.000000Z"
        }
    ]
}
```

#### 6. Get Pending Users (Not Approved)

```http
GET /api/admin/users/pending
Authorization: Bearer {admin_token}
```

**Response:**

```json
{
    "success": true,
    "data": [
        {
            "id": 2,
            "email": "user@example.com",
            "phone": "+9876543210",
            "role": "user",
            "is_approved": false,
            "created_at": "2026-01-31T10:05:00.000000Z"
        }
    ]
}
```

#### 7. Approve User

```http
POST /api/admin/users/{user_id}/approve
Authorization: Bearer {admin_token}
```

**Response:**

```json
{
    "success": true,
    "message": "User approved successfully",
    "data": {
        "id": 2,
        "email": "user@example.com",
        "phone": "+9876543210",
        "role": "user",
        "is_approved": true
    }
}
```

#### 8. Reject/Disapprove User

```http
POST /api/admin/users/{user_id}/reject
Authorization: Bearer {admin_token}
```

**Response:**

```json
{
    "success": true,
    "message": "User rejected successfully",
    "data": {
        "id": 2,
        "email": "user@example.com",
        "phone": "+9876543210",
        "role": "user",
        "is_approved": false
    }
}
```

#### 9. Delete User

```http
DELETE /api/admin/users/{user_id}
Authorization: Bearer {admin_token}
```

**Response:**

```json
{
    "success": true,
    "message": "User deleted successfully"
}
```

### User Endpoints (Require Approved Status)

#### 10. Get User Profile

```http
GET /api/user/profile
Authorization: Bearer {user_token}
```

**Response:**

```json
{
    "success": true,
    "data": {
        "id": 2,
        "email": "user@example.com",
        "phone": "+9876543210",
        "role": "user",
        "is_approved": true,
        "created_at": "2026-01-31T10:05:00.000000Z",
        "updated_at": "2026-01-31T10:10:00.000000Z"
    }
}
```

#### 11. Update User Profile

```http
PUT /api/user/profile
Authorization: Bearer {user_token}
Content-Type: application/json

{
    "phone": "+1111111111",
    "password": "newpassword123",
    "password_confirmation": "newpassword123"
}
```

**Response:**

```json
{
    "success": true,
    "message": "Profile updated successfully",
    "data": {
        "id": 2,
        "email": "user@example.com",
        "phone": "+1111111111",
        "role": "user",
        "is_approved": true
    }
}
```

## Error Responses

### 401 Unauthorized

```json
{
    "message": "Unauthenticated."
}
```

### 403 Forbidden (Admin Access Required)

```json
{
    "success": false,
    "message": "Unauthorized. Admin access required."
}
```

### 403 Forbidden (User Not Approved)

```json
{
    "success": false,
    "message": "Your account is not approved yet. Please wait for admin approval."
}
```

### 404 Not Found

```json
{
    "success": false,
    "message": "Resource not found"
}
```

### 422 Validation Error

```json
{
    "message": "The email field is required.",
    "errors": {
        "email": ["The email field is required."]
    }
}
```

## Database Schema

### Users Table

| Column         | Type       | Description                         |
| -------------- | ---------- | ----------------------------------- |
| id             | bigint     | Primary key                         |
| email          | string     | Unique email address                |
| phone          | string(20) | Unique phone number                 |
| password       | string     | Hashed password                     |
| role           | enum       | 'admin' or 'user' (default: 'user') |
| is_approved    | boolean    | Approval status (default: false)    |
| remember_token | string     | Remember me token                   |
| created_at     | timestamp  | Creation timestamp                  |
| updated_at     | timestamp  | Update timestamp                    |

## Testing with Postman/Insomnia

### 1. Register a new user

- POST `http://localhost:8000/api/register`
- Body: JSON with email, phone, password, password_confirmation

### 2. Login as admin

- POST `http://localhost:8000/api/login`
- Body: `{"email": "admin@voterlist.com", "password": "admin123"}`
- Copy the token from response

### 3. Get pending users

- GET `http://localhost:8000/api/admin/users/pending`
- Header: `Authorization: Bearer {admin_token}`

### 4. Approve the new user

- POST `http://localhost:8000/api/admin/users/{id}/approve`
- Header: `Authorization: Bearer {admin_token}`

### 5. Login as approved user

- POST `http://localhost:8000/api/login`
- Body: `{"email": "registered_email", "password": "user_password"}`

### 6. Access user profile

- GET `http://localhost:8000/api/user/profile`
- Header: `Authorization: Bearer {user_token}`

## Security Features

- ✅ Password hashing using bcrypt
- ✅ Token-based authentication with Laravel Sanctum
- ✅ Role-based middleware protection
- ✅ Approval status verification
- ✅ Input validation
- ✅ Protected admin routes
- ✅ CORS configuration

## Mobile Application Integration

### Token Storage

Store the authentication token received after login in secure storage (Keychain for iOS, Keystore for Android).

### API Calls

Include the token in the Authorization header for all protected endpoints:

```
Authorization: Bearer {token}
```

### User Flow

1. **Registration**: User registers → Receives success message → Wait for approval
2. **Login Attempt**: User tries to login → If not approved, show "waiting for approval" message
3. **After Approval**: User can login successfully → Store token → Access profile
4. **Admin Flow**: Admin logs in → View all users/pending users → Approve/Reject users

## Troubleshooting

### MySQL Connection Error

- Ensure MySQL is running in XAMPP
- Verify database name `voter_list` exists
- Check `.env` database credentials

### 500 Internal Server Error

- Check Laravel logs: `storage/logs/laravel.log`
- Ensure all migrations ran successfully
- Verify file permissions on `storage` and `bootstrap/cache`

### Token Not Working

- Ensure `Authorization: Bearer {token}` header is included
- Check if token is valid (not expired or deleted)
- Verify Sanctum is properly installed

## Development

### Clear Cache

```bash
php artisan cache:clear
php artisan config:clear
php artisan route:clear
```

### Run Tests

```bash
php artisan test
```

### Code Style

```bash
./vendor/bin/pint
```

## License

This project is open-sourced software licensed under the MIT license.

## Support

For issues or questions, please create an issue in the repository.
